<?php

/**
 * Base_Exception
 *
 * Handler of exceptions. Also contains handler of php errors and handler of uncaught exceptions
 *
 * @package System
 * @author Screamer <nwotnbm@gmail.com>
 * @version 0.1
 */

class Base_Exception extends Exception {

    /**
     * Display exceptions: TRUE - on display; FALSE - to log file;
     *
     * @var (boolean)
     * @static
     * @access public
     */
    public static $displayMessage = TRUE;

    /**
     * Path to log file
     *
     * @var (string)
     * @static
     * @access public
     */
    public static $logFile = '';

    /**
     * Contains description of php errors constants
     *
     * @var (array)
     * @static
     * @access protected
     */
    protected static $errorCodes = array (
        E_ERROR => 'Fatal Error',
        E_WARNING => 'Warning',
        E_PARSE => 'Parse error',
        E_NOTICE => 'Notice',
        E_USER_ERROR => 'User error',
        E_USER_WARNING => 'User warning',
        E_USER_NOTICE => 'User notice',
        E_STRICT => 'Strict',
        E_RECOVERABLE_ERROR => 'Recoverable error',
        E_DEPRECATED => 'Deprecated',
        E_USER_DEPRECATED => 'User deprecated'
    );

    /**
     * Constructor
     *
     * @access public
     * @param  (string) $message message of exception
     * @param  (mixed)  $code    code of exception
     * @return (void)
     */
    public function __construct($message, $code = 0)
    {

        parent::__construct($message, (int) $code);
        /*
        Save unmodified error code
        https://bugs.php.net/bug.php?id=39615
        */
        $this->code = $code;

    }

    /**
     * Handle uncaught exceptions
     *
     * used in set_exception_handler()
     *
     * @access public
     * @static
     * @param  (object) $exception object of exception
     * @return (void)
     */
    public static function handler(Exception $exception)
    {

        $type = get_class($exception);
        $message = $exception->getMessage();
        $file = $exception->getFile();
        $line = $exception->getLine();
        $trace = $exception->getTraceAsString();
        $code = $exception->getCode();

        if ($exception instanceof ErrorException)
        {

            $code = self::$errorCodes[$code];

        }

        if (self::$displayMessage)
        {

            echo '<pre><b>' . $type . ' [' . $code . ']</b><br /><b>Message:</b> ' . $message .
             '<br /><b>File:</b> ' . $file .
             '<br /><b>Line:</b> ' . $line .
             '<br /><b>Trace:</b><br />' . $trace . '</pre>';

        }
        else
        {

            /* Write to log */
            $data = '';

            if (file_exists(self::$logFile))
            {

                $data .= file_get_contents(self::$logFile);

            }

            $data .=  PHP_EOL . '(' . date('d:m:Y H:i:s', time()) . ') ' . PHP_EOL .
                     $type . ' [' . $code . ']' . PHP_EOL .
                     'Message: ' . $message . PHP_EOL .
                     'Line: ' . $line . PHP_EOL .
                     'File: ' . $file . PHP_EOL .
                     'Trace: ' . PHP_EOL . $trace . PHP_EOL;
            file_put_contents(self::$logFile, $data);

        }

        exit(1);

    }

    /**
     * Handler of php errors
     *
     * used in set_error_handler()
     *
     * @access public
     * @static
     * @throws Error_Exception
     * @param  (int)     $severity contains the level of the error raised, as an integer.
     * @param  (string)  $message  contains the error message, as a string.
     * @param  (mixed)   $filename contains the filename that the error was raised in, as a string.
     * @param  (mixed)   $line     contains the line number the error was raised at, as an integer.
     * @return (boolean)
     */

    public static function errorHandler($severity, $message, $filename = NULL, $line = NULL)
    {

        if (error_reporting() && $severity)
        {

            throw new ErrorException($message, $severity, 0, $filename, $line);

        }

        return TRUE;

    }

}