<?php

/**
 * Router
 *
 * Load module. Execute requested action.
 *
 * @package System
 * @author Screamer <nwotnbm@gmail.com>
 * @version 0.1
 */

class Router
{

    /**
     * @var (string)
     * Directory of modules
     * @access protected
     */
    protected $Directory;

    /**
     * @var (string)
     * Extension of module
     * @access protected
     */
    protected $Ext;

    /**
     * @var (string)
     * Action of module
     * @access protected
     */
    protected $Action;

    /**
     * @var (string)
     * Default action
     * @access protected
     */
    protected $ActionDefault;

    /**
     * @var (string)
     * Path to module
     * @access protected
     */
    protected $Path;

    /**
     * @var (string)
     * Name of module
     * @access protected
     */
    protected $Name;

    /**
     * @var (string)
     * Error handler
     * @access protected
     */
    protected $Error;

    /**
     * @var (string)
     * Default module
     * @access protected
     */
    protected $Default;

    /**
     * Constructor
     * @access public
     * @param (string) $module name of module
     * @param (string) $action action
     * @param (array)  $params parameters for module
     * @param (object)  $Conf   Router configuration
     * Description of configuration parameter ($Conf)
     * [directory]  (string)  directory of modules
     * [default]    (string)  default module
     * [error]      (string)  error module
     * [ext]        (string)  extension of module
     * @throws Base_Exception
     * @return (void)
     */
    public function __construct($Conf, $module = '', $action = '', array $params)
    {

        /* Checking configuration */
        $keys = array('directory' => '', 'default' => '', 'error' => '', 'ext' => '', 'action' => '');

        foreach($keys as $key => $val)
        {

            if (!empty($Conf->$key))
            {

                unset($keys[$key]);

            }

        }

        if (!empty($keys))
        {

            throw new Base_Exception(
                'Router got wrong parameters.' . PHP_EOL .
                'Please, check the next parameters in your configuration file: ' . PHP_EOL .
                implode(', ', $keys),
                __CLASS__
            );

        }

        /* Set configuration */
        $this->Directory = $Conf->directory . DIRECTORY_SEPARATOR;
        $this->Ext = $Conf->ext;
        $this->Default = $Conf->default;
        $this->ActionDefault = $Conf->action;
        $this->Error = $Conf->error;
        $this->executeModule($module, $action, $params);

    }

    /**
     * Execute module
     * @access protected
     * @param  (string) $module module
     * @param  (string) $action requested action
     * @params (array)  $params parameters for module
     * @throws Base_Exception
     * @return (void)
     */
    protected function executeModule($module = '', $action = '', array $params)
    {

        /* Set default module */
        if (empty($module))
        {

            $module = $this->Default;

        }

        /* Set default action */
        if (empty($action))
        {

            $action = $this->ActionDefault;

        }

        /* Get module */
        $object = $this->fetchObject($this->getPath($module), $module, $action, $params);

        if ($object === FALSE)
        {

            $action = $this->ActionDefault;
            $object = $this->fetchObject($this->getPath($this->Error), $this->Error, $action, $params);

            if ($object === FALSE)
            {

                throw new Base_Exception('Router::Unable to load error handler ' . $this->getPath($this->Error), __CLASS__);

            }

        }

        /* Execute */
        $object->$action();

    }

    /**
     * Get absolute path to module
     * @access (protected)
     * @param  (string)     name of module
     * @return (string)
     */
    protected function getPath($module)
    {

        return $this->Directory . $module . DIRECTORY_SEPARATOR . 'controller' . DIRECTORY_SEPARATOR . $module . $this->Ext;

    }

    /**
     * Get object of module and action
     * If unable to load module, then return FALSE. Otherwise return object.
     * @access protected
     * @param  (string) $path   path to module
     * @param  (string) $class  name of controller class
     * @param  (string) $action method of controller
     * @param  (array)  $params some parameters for module
     * @return (mixed)
     */
    protected function fetchObject($path, $class, $action, array $params)
    {

        if (file_exists($path))
        {

            $class .= '_Controller';
            require_once $path;

            if (class_exists($class))
            {

                $object = new $class($params);
                if (is_callable(array($object, $action)))
                {

                    return $object;

                }

            }

        }

        return FALSE;

    }

}