<?php

/**
 * Db
 *
 * Provides access to the database using a mysqli object
 *
 * @package System
 * @author Screamer <nwotnbm@gmail.com>
 * @version 0.1
 */

class Db {

    /**
     * @var (object)
     * The object which represent a connection to the server
     * @access protected
     */
    protected $mysqli;

    /**
     * Construct
     * Get connection parameters. Connect to mysql server. Setup connection charset
     * @access public
     * @param (object) $conf Connection parameters
     * Description of connection parameters
     * [host]     (string) hostname
     * [user]     (string) username
     * [password] (string) password
     * [db]       (string) database
     * [charset]  (string) charset
     * @return (void)
     */

    public function __construct($conf)
    {

        $host = isset($conf->host) ? $conf->host : 'localhost';
        $user = isset($conf->user) ? $conf->user : 'root';
        $password = isset($conf->password) ? $conf->password : '';
        $db = isset($conf->db) ? $conf->db : 'test';
        $charset = isset($conf->charset) ? $conf->charset : 'utf8';

        try
        {

            $this->mysqli = new mysqli($host, $user, $password, $db);

        }
        catch (Exception $e)
        {

            throw new Base_Exception($e, __CLASS__);

        }

        /* Setup charset */
        $this->query("SET NAMES " . $charset);

    }

    /**
     * Close connection
     * @access public
     * @return (boolean)
     */

    public function close()
    {

        return $this->mysqli->close();

    }

    /**
     * Change database
     * @access public
     * @param (string) $name name of database
     * @return (boolean)
     */

    public function selectDb($name)
    {

        return $this->mysqli->select_db($name);

    }

    /**
     * Performs a query on the database
     * Returns FALSE on failure. For successful SELECT, SHOW, DESCRIBE or EXPLAIN queries mysqli_query()
     * will return a mysqli_result object. For other successful queries mysqli_query() will return TRUE.
     * @access public
     * @param  (string) $string     query string
     * @param  (int)    $resulttype type of result. MYSQLI_USE_RESULT – unbuffered. MYSQL_STORE_RESULT - buffered
     * @throws Base_Exception
     * @return (mixed)
     */

    public function query($string, $resulttype = MYSQLI_USE_RESULT)
    {

        $query = $this->mysqli->query($string, $resulttype);

        if ($query === FALSE)
        {

            throw new Base_Exception($this->mysqli->error, $this->mysqli->errno);

        }

        return $query;

    }

    /**
     * Escapes special characters in a string for use in an SQL statement, taking into account the current charset of the connection
     * @access public
     * @param  (string) $string The string to be escaped. Characters encoded are NUL (ASCII 0), \n, \r, \, ', ", and Control-Z.
     * @return (string)
     */

    public function escapeString($string)
    {

        return $this->mysqli->real_escape_string($string);

    }

    /**
     * Fetches all result rows as an associative array, a numeric array, or both. Доступно только с расширением mysqlnd.
     * Returns an array of associative or numeric arrays holding result rows.
     * @access public
     * @param  (mysqli_result) $result     Result of query
     * @param  (int)           $resulttype This optional parameter is a constant indicating what type of array should be produced
     * from the current row data. The possible values for this parameter
     * are the constants MYSQLI_ASSOC, MYSQLI_NUM, or MYSQLI_BOTH.
     * @return (array)
     */

    public function fetchAll(mysqli_result $result, $resulttype = MYSQLI_NUM)
    {

        return $result->fetch_all($resulttype);

    }

    /**
     * Fetches result string as an associative array.
     * @access public
     * @param  (mysqli_result) $result result of query
     * @return (array)
     */

    public function fetchAssoc(mysqli_result $result)
    {

        return $result->fetch_assoc();

    }

    /**
     * Get a result row as an enumerated array
     * @access public
     * @param (mysqli_result) $result result of query
     * @return (mixed)
     */

    public function fetchRow(mysqli_result $result)
    {

        return $result->fetch_row();

    }

    /**
     * Returns the next field in the result set
     * @access public
     * @param (mysqli_result) $result result of query
     * @return (object)
     */

    public function fetchField(mysqli_result $result)
    {

        return $result->fetch_field();

    }

    /**
     * Frees the memory associated with a result
     * You should always free your result with freeResult(), when your result object is not needed anymore.
     * @access public
     * @param (mysqli_result) $result result of query
     * @return (void)
     */

    public function freeResult(mysqli_result $result)
    {

        $result->free_result();

    }

    /**
     * Analog mysql_result()
     * @access public
     * @param (mysqli_result) $result result of query
     * @return (string)
     */

    public function result(mysqli_result $result)
    {

        $result = $this->fetchRow($result);
        return $result[0];

    }

    /**
     * Get number of rows in the result set
     * @access public
     * @param (mysqli_result) $result result of query
     * @return (int)
     */

    public function numRows(mysqli_result $result)
    {

        return $result->num_rows();

    }

    /**
     * Gets the automatically generated ID, using recent request
     * @access public
     * @return (mixed)
     */

    public function insertId()
    {

        return $this->mysqli->insert_id;

    }

    /**
     * Returns the number of rows affected by last query
     * @access public
     * @return int
     */

    public function affectedRows()
    {

        return $this->mysqli->affected_rows;

    }

}