<?php

/**
 * Templates handler
 *
 * Loading template default. Loading templates of module, macroses and data
 * for default template. Create main template.
 * Create object of handler:
 * $Tpl = new Template($configuration);
 * Description of configuration
 * [directory] (string) Absolute path to default directory of templates. You can use dot as directory separator.
 * [template]  (string) Main template
 * [ext]       (string) Extension of templates with dot (Example: .php)
 *
 * Set path to templates of module:
 * $Tpl->Path = 'YourPath';
 * Sample of loading  template of module:
 * $Tpl->setTemplate('nameOfTemplate', 'templateFile', array('someData'));
 * Sample of macroses assigning:
 * $Tpl->setMacros('somefunc', function($param1, $param2) { return '<h1>' . $param1 . '</h1><h2>' . $param2 . '</h2>'; }
 * Sample of adding data:
 * $Tpl->setData('pageTitle', 'Title');
 * Now you can render data for output...
 * $data = $Tpl->render();
 * ... and display it:
 * echo $data;
 *
 * @package System
 * @author Screamer <nwotnbm@gmail.com>
 * @version 0.1
 */

class Template {

    /**
     * @var (string)
     * @access public
     * Path to templates of module
     */
    public $Path;

    /**
     * @var (string)
     * @access protected
     * Main template
     */
    protected $DefaultTemplate;

    /**
     * @var (string)
     * @access protected
     * Extension of templates (with dot: .php)
     */
    protected $Ext;

    /**
     * @var (string)
     * @access protected
     * This storage of templates
     */
    protected $Templates = array();

    /**
     * @var (string)
     * @access protected
     * This storage of data for default template
     */
    protected $Data = array();

    /**
     * @var (string)
     * @access protected
     * Location for macroses
     */
    protected $Macroses = array();

    /**
     * Constructor
     * Load configuration. Checking default template.
     * @access public
     * @param  (object) $conf Configuration
     * Description of configuration
     * [directory] (string) Absolute path to default directory of templates. You can use dot as directory separator.
     * [template]  (string) Main template
     * [ext]       (string) Extension of templates with dot (Example: .php)
     * @throws Base_Exception
     * @return (void)
     */
    public function __construct($conf)
    {

        /* Checking configuration */
        if (empty($conf->directory) || empty($conf->template) || empty($conf->ext))
        {

            $message = 'incorrect parameters:' . PHP_EOL .
                       'directory => ' . (isset($conf->directory) ? $conf->directory : '') . PHP_EOL .
                       'template => ' . (isset($conf->template) ? $conf->template : '') . PHP_EOL .
                       'ext => ' . (isset($conf->ext) ? $conf->ext : '');
            throw new Base_Exception($message, __CLASS__);

        }

        $this->Ext = $conf->ext;
        $this->DefaultTemplate = str_replace('.', DIRECTORY_SEPARATOR, $conf->directory . '.') . $conf->template . $this->Ext;

        /* Checking default template */
        if (!file_exists($this->DefaultTemplate))
        {

            throw new Base_Exception('Main template ' . $this->DefaultTemplate . ' is not found', __CLASS__);

        }

    }

    /**
     * Add data for default template
     * @access public
     * @param (string) $name name of variable
     * @param (mixed)  $value value of variable
     * @return (void)
     */
    public function setData($name, $value)
    {

        $this->checkExists($name);
        $this->Data[$name] = $value;

    }

    /**
     * Add template
     * @access public
     * @param (string) $name name of template in main template
     * @param (string) $template name of template file
     * @param (array)  $data data for template
     * @throws Base_Exception
     * @return (void)
     */
    public function setTemplate($name, $template, array $data = array())
    {

        if (!file_exists($this->Path . $template . $this->Ext))
        {

            throw new Base_Exception('Template ' . $this->Path . $template .  $this->Ext . ' is not exists', __CLASS__);

        }

        $this->checkExists($name);

        $this->Templates[$name] = array('template' => $template, 'data' => $data);

    }


    /**
     * Set macros
     * @access public
     * @param (string) $name name of macros
     * @param (Closure) $function lambda function
     * @return (void)
     */
    public function setMacros($name, Closure $function)
    {

        $this->checkExists($name);
        $this->Macroses[$name] = $function;

    }

    /**
     * Check object existence
     * @access protected
     * @param (string) $name name of object
     * @throws Base_Exception
     * @return(void)
     */
    protected function checkExists($name)
    {

        $data = !empty($this->Data[$name]) ? PHP_EOL . __CLASS__ . '::Data[' . $name . '] => ' . $this->Data[$name] : '';
        $macroses = !empty($this->Macroses[$name]) ? PHP_EOL . __CLASS__ . '::Macroses[' . $name . '] => ' . $this->Macroses[$name] : '';
        $templates = !empty($this->Templates[$name]) ? PHP_EOL . __CLASS__ . '::Templates[' . $name . '] =>' . $this->Templates[$name] : '';
        $message = 'Unable to set ' . $name . '.' . $data . $macroses . $templates;

        if (!empty($data) || !empty($macroses) || !empty($templates))
        {

            throw new Base_Exception($message, __CLASS__);

        }

    }

    /**
     * Capture template
     * @access protected
     * @param (string) $template path to template
     * @param (array) $data data for template
     * @return (string)
     */
    protected function capture($template, $data = array())
    {

        if (!empty($data))
        {

            extract($data, EXTR_SKIP);

        }

        ob_start();
        include_once($template);
        $contents = ob_get_contents();
        ob_end_clean();
        return $contents;

    }

    /**
     * Calling macros
     * @access public
     * @param (string) $name name of macros
     * @param (array)  $args arguments
     * @throws Base_Exception
     * @return (mixed)
     */

    public function __call($name, $args)
    {

        if (!isset($this->Macroses[$name]))
        {

            throw new Base_Exception('Macros ' . $name . ' is not exists', __CLASS__);

        }

        return call_user_func_array($this->Macroses[$name], $args);

    }

    /**
     * Create template for output
     * @access public
     * @return (string)
     */
    public function render()
    {

        /* Assign templates */
        if (!empty($this->Templates))
        {

            $temp = array();

            foreach ($this->Templates as $name => $data)
            {

                $temp[$name] = $this->capture($this->Path . $data['template'] . $this->Ext, $data['data']);

            }

            $this->Data = array_merge($this->Data, $temp);
            unset($temp, $this->Templates);

        }

        return $this->capture($this->DefaultTemplate, $this->Data);

    }

}