<?php

/**
 * JLayer
 * JohnCMS core wrapper
 * @package JohnCMS
 * @subpackage JLayer
 * @author Screamer
 * @copyright 2013
 */

class JLayer {

	/**
	 * @var IP address
	 */
	private $ip;

	/**
	 * @var IP address via PROXY server
	 */
	private $ip_via_proxy;

	/**
	 * @var User-Agent
	 */
	private $user_agent;

	/**
	 * @var Is mobile browser?
	 */
	private $is_mobile;

	/**
	 * @var System settings
	 */
	private $system_settings;

	/**
	 * @var User Settings
	 */
	private $user_settings;

	/**
	 * @var Data of user
	 */
	private $user_data;

	/**
	 * @var Infringements of user
	 */
	private $user_ban;

	/**
	 * @var Prefix for database tables
	 */
	private $table_prefix;

	/**
	 * @var Path to directory with Models
	 */
	private $models_dir;

	/**
	 * @var Loaded models
	 */
	private $models = array();

	/**
	 * Construct
	 * @param (int) IP address
	 * @param (int) IP address via PROXY server
	 * @param (string) User-Agent
	 * @param (boolean) Is mobile browser?
	 * @param (array) System settings
	 * @param (array) User Settings
	 * @param (array) Data of user
	 * @param (array) Infringements of user
	 * @param (string) Prefix for database tables
	 * @param (string) Path to directory with Models
	 * @return (void)
	 */
	public function __construct(
		$ip,
		$ip_via_proxy,
		$user_agent,
		$is_mobile,
		array $system_settings,
		array $user_settings,
		array $user_data,
		array $user_ban,
		$table_prefix,
		$models_dir
	) {
		$this->ip = intval($ip);
		$this->ip_via_proxy = intval($ip_via_proxy);
		$this->user_agent = trim($user_agent);
		$this->is_mobile = (bool) $is_mobile;
		$this->system_settings = $system_settings;
		$this->user_settings = $user_settings;
		$this->user_data = $user_data;
		$this->user_ban = $user_ban;
		$this->table_prefix = trim($table_prefix);
		if (empty($this->table_prefix)) {
			throw new Exception('Invalid value of tables prefix');
		}
		if (!is_dir($models_dir)) {
			throw new Exception('Wrong path to directory of models: ' . $models_dir);
		}
		$this->models_dir = $models_dir;
	}

	/**
	 * Get some data
	 * @param (string) $type Type of data: 'system_settings', 'user_settings', 'user_data', 'user_ban'
	 * @param (string) $key Name of Item (if empty, returns all array)
	 * @return (mixed)
	 */
	public function get_data($type, $key = '') {
		if (!is_string($key)) {
			throw new Exception('Invalid type of second argument. String expect.');
		}
		$types = array('system_settings', 'user_settings', 'user_data', 'user_ban');
		if (in_array($type, $types)) {
			$data = $this->$type;
		} else {
			throw new Exception('Invalid value of first argument. Following values are allowed: ' . implode(', ', $types));
		}
		if (!empty($key)) {
			if (!array_key_exists($key, $data)) {
				throw new Exception('Unable to get system settings item. Key "' . $key . '" is not exists.');
			}
			return $data[$key];
		} else {
			return $data;
		}
	}

	/**
	 * Returns IP address
	 * @param (boolean) Format address?
	 * @return (int|string)
	 */
	public function get_ip($format = false) {
		return $format ? long2ip($this->ip) : $this->ip;
	}

	/**
	 * Returns IP address via PROXY server
	 * @param (boolean) Format address?
	 * @return (int|string)
	 */
	public function get_ip_via_proxy($format = false) {
		return $format ? long2ip($this->ip_via_proxy) : $this->ip_via_proxy;
	}

	/**
	 * Returns User-Agent
	 * @return (string)
	 */
	public function get_user_agent() {
		return $this->user_agent;
	}

	/**
	 * Is browser mobile?
	 */
	public function is_mobile() {
		return $this->is_mobile;
	}

	/**
	 * Load Model
	 * @param (string) $name Name of model
	 * @return (object)
	 */
	public function model($name) {
		if (!is_string($name)) {
			throw new Exception('Invalid type of argument. String expect.');
		}
		if (!array_key_exists($name, $this->models)) {
			$model = $this->models_dir . $name . '.php';
			if (is_file($model)) {
				require $model;
				$class = 'Model_' . ucwords($name);
				if (class_exists($class)) {
					$this->models[$name] = new $class(new JL_MySQL, $this->table_prefix);
				} else {
					throw new Exception('Unable to load Model. Class ' . $class . ' is not exists.');
				}
			} else {
				throw new Exception('Unable to load Model. File ' . $model . ' is not exists.');
			}
		}
		return $this->models[$name];
	}

}