<?php
namespace jl {

	use jl\exception\InvalidArgumentException;

	/**
	 * JLayer
	 *
	 * Models parent
	 *
	 * @package JohnCMS
	 * @subpackage JLayer
	 * @author Screamer
	 * @copyright 2013
	 */

	abstract class Model {

		/**
		 * @var \jl\Db Mysql wrapper
		 */
		protected $db;

		/**
		 * @var string Prefix for tables
		 */
		private $table_prefix;

		/**
		 * @var string Name of table in database
		 */
		protected $table_name;

		/**
		 * Construct
		 *
		 * @param Db     $handler Object of MySQL wrapper
		 * @param string $prefix  Prefix for tables
		 *
		 * @throws exception\InvalidArgumentException
		 * @return \jl\Model
		 */
		public function __construct(Db $handler, $prefix) {
			if (!is_string($prefix)) {
				throw new InvalidArgumentException('prefix', 'string');
			}

			$this->db = $handler;
			$this->table_prefix = trim($prefix);
		}

		/**
		 * Set name of table
		 *
		 * @param string $name
		 *
		 * @throws exception\InvalidArgumentException
		 * @return void
		 */
		public function set_table_name($name) {
			if (!is_string($name)) {
				throw new InvalidArgumentException('name', 'string');
			}

			$this->table_name = $this->table_prefix . '_' . $name;
		}

		/**
		 * Get data of some object
		 *
		 * @param int|string $id ID of item
		 * @param string     $by Field which contains object identifier
		 *
		 * @throws exception\InvalidArgumentException
		 * @return array
		 */
		public function get($id, $by) {
			if (!is_string($by)) {
				throw new InvalidArgumentException('by', 'string');
			}

			$id = is_string($id) ? $this->db->escape_string($id) : intval($id);
			$result = $this->db->query("SELECT * FROM `" . $this->table_name . "` WHERE `" . $by . "` = '" . $id . "'");
			$data = $this->db->fetch_assoc($result);
			$this->db->free($result);
			return is_array($data) ? $data : array();
		}

		/**
		 * Get list of some objects
		 *
		 * @param string|int|null $id     ID of object
		 * @param string|null     $by     Field which contains object identifier
		 * @param string          $order  Order rules
		 * @param int             $start  Start position
		 * @param int             $end    End position
		 *
		 * @throws exception\InvalidArgumentException
		 * @return array
		 */
		public function get_list($id = null, $by = null, $order, $start = 0, $end = 0) {
			if (!is_string($id) && !is_int($id) && !is_null($id)) {
				throw new InvalidArgumentException('id', 'string, integer, null');
			}
			if (!is_string($by) && !is_null($by)) {
				throw new InvalidArgumentException('by', 'string, null');
			}
			if (!is_string($order)) {
				throw new InvalidArgumentException('order_fields', 'string');
			}
			if (!is_int($start)) {
				throw new InvalidArgumentException('start', 'integer');
			}
			if (!is_int($end)) {
				throw new InvalidArgumentException('end', 'integer');
			}

			$statement = "SELECT * FROM `" . $this->table_name . "`";
			if (!is_null($id)) {
				$id = is_string($id) ? $this->db->escape_string($id) : intval($id);
				$statement .= " WHERE `" . $by . "` = '" . $id . "'";
			}
			if (!empty($order)) {
				$statement .= " ORDER BY " .$order;
			}
			if ($end > 0) {
				$statement .= " LIMIT " . $start . $end;
			}
			$result = $this->db->query($statement);
			$list = array();
			while ($item = $this->db->fetch_assoc($result)) {
				$list[] = $item;
			}
			$this->db->free($result);
			return $list;
		}

		/**
		 * Save data of some object
		 *
		 * @param array           $data Data for save
		 * @param string|int|null $id   ID of object
		 * @param string|null     $by   Field which contains object identifiers
		 *
		 * @throws exception\InvalidArgumentException
		 * @return int|string
		 */
		public function save(array $data, $id = null, $by = null) {
			if (!is_string($id) && !is_int($id) && !is_null($id)) {
				throw new InvalidArgumentException('id', 'string, integer, null');
			}
			if (!is_string($by) && !is_null($by)) {
				throw new InvalidArgumentException('by', 'string, null');
			}

			if (!is_null($id)) {
				$id = is_string($id) ? $this->db->escape_string($id) : intval($id);
				$statement = "UPDATE `" . $this->table_name . "` SET ";
			} else {
				$statement = "INSERT INTO `" . $this->table_name . "` SET ";
			}
			foreach ($data as $field => $value) {
				$statement .= "`" . $field . "` = '" . (is_string($value) ? $this->db->escape_string($value) : intval($value)) . "',";
			}
			$statement = rtrim($statement, ",") . (is_null($id) ? "" : " WHERE `" . $by . "` = '" . $id . "'");
			$this->db->query($statement);
			return is_null($id) ? $this->db->insert_id() : $id;
		}

		/**
		 * Remove some item by ID
		 *
		 * @param int|string $id ID of item
		 * @param string     $by Field which contains object identifier
		 *
		 * @throws exception\InvalidArgumentException
		 * @return int
		 */
		public function remove($id, $by) {
			if (!is_string($by)) {
				throw new InvalidArgumentException('by', 'string');
			}

			$id = is_string($id) ? $this->db->escape_string($id) : intval($id);
			$this->db->query("DELETE FROM `" . $this->table_name . "` WHERE `" . $by . "` = '" . $id . "'");
			return $this->db->affected_rows();
		}

	}
}