<?php
namespace jl\plugins {

	use jl\exception\Exception;
	use jl\exception\InvalidArgumentException;
	use jl\Plugin;

	/**
	 * JLayer
	 *
	 * Images handler.
	 * GD only required.
	 *
	 * @package JohnCMS
	 * @subpackage JLayer
	 * @author Screamer
	 * @copyright 2013
	 */
	class Image extends Plugin {

		/**
		 * int Width for resize image by default
		 */
		const p_width = 240;

		/**
		 * int Height for resize image by default
		 */
		const p_height = 320;

		/**
		 * @var resource Image gd resource
		 */
		private $image;

		/**
		 * @var int Width of image
		 */
		private $width;

		/**
		 * @var int Height of image
		 */
		private $height;

		/**
		 * @var int Type of image
		 */
		private $type;

		/**
		 * @var string MIME-Type of image
		 */
		private $mime;

		/**
		 * Load image from file
		 *
		 * @param string $path Path to image
		 *
		 * @return boolean
		 */
		public function load_image($path) {
			if (is_file($path)) {
				if (is_resource($this->image)) {
					imagedestroy($this->image);
				}
				$this->image = imagecreatefromstring(file_get_contents($path));
				$info = getimagesize($path);
				$this->width = $info[0];
				$this->height = $info[1];
				$this->type = $info[2];
				$this->mime = $info['mime'];
				return true;
			} else {
				return false;
			}
		}

		/**
		 * Resize image
		 *
		 * @param int $width  New width in pixels
		 * @param int $height New height in pixels
		 *
		 * @throws \jl\exception\Exception
		 * @throws \jl\exception\InvalidArgumentException
		 * @return resource
		 */
		public function resize($width = 0, $height = 0) {
			if (!is_resource($this->image)) {
				throw new Exception('Image is not loaded.');
			}
			if (!is_int($width)) {
				throw new InvalidArgumentException('width', 'int');
			}
			if (!is_int($height)) {
				throw new InvalidArgumentException('height', 'int');
			}

			if ($width <= 0) {
				$width = self::p_width;
			}
			if ($height <= 0) {
				$height = self::p_height;
			}

			// Get image width and height
			$image_width = imagesx($this->image);
			$image_height = imagesy($this->image);

			// Calculate new width and height
			if ($image_width > $width || $image_height > $height) {
				$x_ratio     = 200 / $image_width;
				$y_ratio     = 320 / $image_height;
				$ratio       = min($x_ratio, $y_ratio);
				$use_x_ratio = ($x_ratio == $ratio);
				$new_width   = $use_x_ratio  ? 200  : floor($image_width * $ratio);
				$new_height  = !$use_x_ratio ? 320 : floor($image_height * $ratio);
			} else {
				$new_width = $image_width;
				$new_height = $image_height;
			}

			// Create resource for preview
			$image_create = 'imagecreate' . ($this->type == IMAGETYPE_GIF ? '' : 'truecolor');
			$image_dest  = $image_create($new_width, $new_height);

			// Save transparent
			if ($this->type == IMAGETYPE_GIF) {
				imagecolortransparent($image_dest, imagecolorallocate($image_dest, 0, 0, 0));
			} elseif ($this->type == IMAGETYPE_PNG) {
				imagealphablending($image_dest, false);
				imagesavealpha($image_dest, true);
			}

			// Resize
			imagecopyresampled(
				$image_dest, $this->image, 0, 0, 0, 0,
				$new_width, $new_height, $image_width, $image_height
			);

			return $image_dest;
		}

		/**
		 * Output image
		 *
		 * @param resource|null   $image Image for output
		 * @param string          $path  Path for output
		 *
		 * @throws \jl\exception\Exception
		 * @throws \jl\exception\InvalidArgumentException
		 * @return void
		 */
		public function output($image = null, $path = '') {
			if (!is_null($image) && !is_resource($image)) {
				throw new InvalidArgumentException('image', 'null, resource');
			}
			if (!is_string($path)) {
				throw new InvalidArgumentException('path', 'string');
			}

			if (!empty($path) && !is_dir($path)) {
				throw new Exception('Unable to output image. Directory "' . $path . '" is not exists.');
			}

			switch($this->type) {
				case IMAGETYPE_GIF:
					$type = 'gif';
					break;
				case IMAGETYPE_JPEG2000:
				case IMAGETYPE_JPEG:
					$type = 'jpeg';
					break;
				case IMAGETYPE_PNG:
					$type = 'png';
					break;
				default:
					throw new Exception('Unable to output image. Wrong type given: ' . $this->type . '.');
			}
			$output = 'image' . $type;
			header('Content-type: ' . $this->mime);
			$output((is_null($image) ? $this->image : $image), (!empty($path) ? $path : null));
		}

	}
}