<?php

class Query
{
    /**
     * @var string
     */
    protected $table;

    /**
     * @var array
     */
    protected $columns = [];

    /**
     * @var array
     */
    protected $joins = [];

    /**
     * @var array
     */
    protected $wheres = [];

    /**
     * @var array
     */
    protected $havings = [];

    /**
     * @var array
     */
    protected $groupings = [];

    /**
     * @var array
     */
    protected $orderings = [];

    /**
     * @var int
     */
    protected $limit;

    /**
     * @var int
     */
    protected $offset;

    /**
     * @var array
     */
    protected $bindings = [];

    /**
     * 
     * 
     * @param string|array $table
     * @return void
     */
    public function __construct($table)
    {
        $this->table = $table;
    }

    /**
     * 
     * 
     * @param string|array $column 
     * @return self
     */
    public function column($column = '*')
    {
        $this->columns[] = $column;

        return $this;
    }

    /**
     * 
     * 
     * @param string $table
     * @param string $condition
     * @param string $type
     * @return self
     */
    public function rawJoin($table, $condition, $type = 'INNER')
    {
        $this->joins[$type][$table] = $condition;

        return $this;
    }

    /**
     * 
     * 
     * @param string $table
     * @param string|Closure $column
     * @param string $comparison
     * @param string $column2
     * @param string $type
     * @return self
     */
    public function join($table, $column, $comparison = null, $column2 = null, $type = 'INNER')
    {
        if ($column instanceOf Closure) {
            $column($nest = new static($this->table));

            return $this->rawJoin($table, $nest, $comparison ?: $type);
        } else {
            return $this->rawJoin($table, compact('column', 'comparison', 'column2'), $type);
        }
    }

    /**
     * 
     * 
     * @param string $column
     * @param string $comparison
     * @param string $column2
     * @return self
     */
    public function on($column, $comparison, $column2)
    {
        return $this->where($column, $comparison, new Expression($this->quoteColumn($column2)));
    }

    /**
     * 
     * 
     * @param string $column
     * @param string $comparison
     * @param string $column2
     * @return self
     */
    public function orOn($column, $comparison, $column2)
    {
        return $this->where($column, $comparison, new Expression($this->quoteColumn($column2)), 'OR');
    }

    /**
     * 
     * 
     * @param string|self $condition
     * @param mixed $values
     * @param string $logic
     * @return self
     */
    public function rawWhere($condition, $values = null, $logic = 'AND')
    {
        $this->bindings = array_merge($this->bindings, (array) $values);
        $this->wheres[] = [$logic => $condition];

        return $this;
    }

    /**
     * 
     * 
     * @param string|self $condition
     * @param mixed $values
     * @return self
     */
    public function rawOrWhere($condition, $values = null)
    {
        return $this->rawWhere($condition, $values, 'OR');
    }

    /**
     * 
     * 
     * @param string|Closure $column
     * @param string $comparison
     * @param mixed $value
     * @param string $logic
     * @return self
     */
    public function where($column, $comparison = '=', $value = null, $logic = 'AND')
    {
        if ($column instanceOf Closure) {
            $column($nest = new static($this->table));

            return $this->rawWhere($nest, $nest->bindings, $logic);
        } else {
            return $this->rawWhere(compact('column', 'comparison', 'value'), $value, $logic);
        }
    }

    /**
     * 
     * 
     * @param string|self $column
     * @param string $comparison
     * @param mixed $value
     * @return self
     */
    public function orWhere($column, $comparison = '=', $value = null)
    {
        return $this->where($column, $comparison, $value, 'OR');
    }

    /**
     * 
     * 
     * @param string $column
     * @return self
     */
    public function group($column)
    {
        $this->groupings[] = $column;

        return $this;
    }

    /**
     * 
     * 
     * @param string $condition
     * @param mixed $values
     * @param string $logic
     * @return self
     */
    public function rawHaving($condition, $values = null, $logic = 'AND')
    {
        $this->bindings  = array_merge($this->bindings, (array) $values);
        $this->havings[] = [$logic => $condition];

        return $this;
    }

    /*
     * 
     * 
     * @param string $condition
     * @param mixed $values
     * @return self
     */
    public function rawOrHaving($condition, $values = null)
    {
        return $this->rawHaving($condition, $values, 'OR');
    }

    /**
     * 
     * 
     * @param string|Closure $column
     * @param string $comparison
     * @param mixed $value
     * @param string $logic
     * @return self
     */
    public function having($column, $comparison = '=', $value = null, $logic = 'AND')
    {
        if ($column instanceOf Closure) {
            $column($nest = new static($this->table));

            return $this->rawHaving($nest, $nest->bindings, $logic);
        } else {
            return $this->rawHaving(compact('column', 'comparison', 'value'), $value, $logic);
        }
    }

    /**
     * 
     * 
     * @param string $column
     * @param string $comparison
     * @param mixed $value
     * @return self
     */
    public function orHaving($column, $comparison = '=', $value = null)
    {
        return $this->having($column, $comparison, $value, 'OR');
    }

    /**
     * 
     * 
     * @param string $column
     * @param string $direction
     * @return self
     */
    public function order($column, $direction = 'ASC')
    {
        $this->orderings[] = compact('column', 'direction');

        return $this;
    }

    /**
     * 
     * 
     * @param int $limit
     * @param int $offset
     * @return self
     */
    public function limit($limit, $offset = 0)
    {
        $this->limit  = (int) $limit;
        $this->offset = (int) $offset;

        return $this;
    }

    // ...

    /**
     * 
     * 
     * @param string $value
     * @return string
     */
    public function quote($value)
    {
        return '`'. str_replace('.', '`.`', $value) .'`';
    }

    /**
     * 
     * 
     * @param mixed $table
     * @return string
     */
    public function quoteTable($table)
    {
        if ($table instanceOf Expression) {
            return (string) $table;
        }

        if (is_array($table)) {
            return $this->quoteTable($table[0]) .' AS '. $this->quoteTable($table[1]);
        } else {
            return $this->quote($table);
        }
    }

    /**
     * 
     * 
     * @param mixed $column
     * @return string
     */
    public function quoteColumn($column)
    {
        return $column == '*' ? $column : $this->quoteTable($column);
    }

    /**
     * 
     * 
     * @param array $columns
     * @return string
     */
    public function quoteColumns(array $columns)
    {
        return join(', ', array_map(array($this, 'quoteColumn'), $columns));
    }

    /**
     * 
     * 
     * @param mixed $value
     * @return string
     */
    public function escapeValue($value)
    {
        if ($value instanceOf Expression) {
            return (string) $value;
        } else {
            return '?';
        }
    }

    /**
     * 
     * 
     * @param array $values
     * @return string
     */
    public function escapeValues(array $values)
    {
        return join(', ', array_map(array($this, 'escapeValue'), $values));
    }

    // ...

    /**
     * 
     * 
     * @param int $offset
     * @return string
     */
    public function offset($offset)
    {
        $this->offset = (int) $offset;

        return $this;
    }

    /**
     * 
     * 
     * @param array $columns
     * @return string
     */
    public function select(array $columns = ['*'])
    {
        if (empty($this->columns)) {
            $this->columns = $columns;
        }

        return $this->buildSql('SELECT '. $this->quoteColumns($this->columns) .' FROM '. $this->quoteTable($this->table));
    }

    /**
     * 
     * 
     * @param array $data
     * @return string
     */
    public function insert(array $data)
    {
        return 'INSERT INTO '. $this->quoteTable($this->table)
            .' ('. $this->quoteColumns(array_keys($data)) .') VALUES ('. $this->escapeValues($data) .')';
    }

    /**
     * 
     * 
     * @param array $data
     * @return string
     */
    public function update(array $data)
    {
        $sets = [];

        foreach ($data as $column => $value) {
            $sets[] = $this->quoteColumn($column) .' = '. $this->escapeValue($value);
        }

        return $this->buildSql('UPDATE '. $this->quoteTable($this->table) .' SET '. join(', ', $sets));
    }

    /**
     * 
     * 
     * @return string
     */
    public function delete()
    {
        return $this->buildSql('DELETE FROM '. $this->quoteTable($this->table));
    }

    /**
     * 
     * 
     * @param string $column 
     * @return string
     */
    public function count($column = '*')
    {
        return $this->aggregation(__FUNCTION__, $column);
    }

    /**
     * 
     * 
     * @param string $column 
     * @return string
     */
    public function max($column)
    {
        return $this->aggregation(__FUNCTION__, $column);
    }

    /**
     * 
     * 
     * @param string $column 
     * @return string
     */
    public function min($column)
    {
        return $this->aggregation(__FUNCTION__, $column);
    }

    /**
     * 
     * 
     * @param string $column 
     * @return string
     */
    public function sum($column)
    {
        return $this->aggregation(__FUNCTION__, $column);
    }

    /**
     * 
     * 
     * @param string $column 
     * @return string
     */
    public function avg($column)
    {
        return $this->aggregation(__FUNCTION__, $column);
    }

    /**
     * 
     * 
     * @param string $function 
     * @param string $column 
     * @return string
     */
    public function aggregation($function, $column)
    {
        $expression = new Expression(strtoupper($function) .'('. $this->quoteColumn($column) .')');

        return $this->select([[$expression, 'aggregation']]);
    }

    /**
     * 
     * 
     * @param string $column
     * @param int $amount
     * @return string
     */
    public function increment($column, $amount = 1)
    {
        $expression = new Expression($this->quoteColumn($column) .' + '. $amount);

        return $this->update([$column => $expression]);
    }

    /**
     * 
     * 
     * @param string $column
     * @param int $amount
     * @return string
     */
    public function decrement($column, $amount = 1)
    {
        $expression = new Expression($this->quoteColumn($column) .' - '. $amount);

        return $this->update([$column => $expression]);
    }

    // ...

    /**
     * 
     * 
     * @param string $sql 
     * @return string
     */
    protected function buildSqlJoins($sql)
    {
        foreach ($this->joins as $type => $join) {
            foreach ($join as $table => $condition) {
                $sql .= ' '. strtoupper($type) .' JOIN '. $this->quoteTable($table);

                if ($condition instanceOf static) {
                    $sql .= ' ON ('. $condition->buildSqlConditions('wheres') .') ';
                } elseif (is_array($condition)) { extract($condition);
                    $sql .= ' ON ('. $this->quoteColumn($column) .' '. $comparison .' '. $this->quoteColumn($column2) .') ';
                } elseif (is_string($condition)) {
                    $sql .= ' '. $condition .' ';
                }
            }
        }

        return $sql;
    }

    /**
     * 
     * 
     * @param string $sql 
     * @return string
     */
    protected function buildSqlWheres($sql)
    {
        return $sql .' WHERE '. $this->buildSqlConditions('wheres');
    }

    /**
     * 
     * 
     * @param string $sql 
     * @return string
     */
    protected function buildSqlGroupings($sql)
    {
        return $sql .' GROUP BY '. $this->quoteColumns($this->groupings);
    }

    /**
     * 
     * 
     * @param string $sql 
     * @return string
     */
    protected function buildSqlHavings($sql)
    {
        return $sql .' HAVING '. $this->buildSqlConditions('havings');
    }

    /**
     * 
     * 
     * @param string $sql 
     * @return string
     */
    protected function buildSqlOrderings($sql)
    {
        $orderings = [];

        foreach ($this->orderings as $ordering) {
            $orderings[] = $this->quoteColumn($ordering['column']) .' '. strtoupper($ordering['direction']);
        }

        return $sql .' ORDER BY '. join(', ', $orderings);
    }

    /**
     * 
     * 
     * @param string $sql 
     * @return string
     */
    protected function buildSqlLimit($sql)
    {
        return $sql .' LIMIT '. $this->limit;
    }

    /**
     * 
     * 
     * @param string $sql 
     * @return string
     */
    protected function buildSqlOffset($sql)
    {
        return $sql .' OFFSET '. $this->offset;
    }

    /**
     *
     *
     * @param string $type
     * @return string
     */
    protected function buildSqlConditions($type, $valueAsColumn = false)
    {
        $sql = '';

        foreach ($this->{$type} as $segment) {
            foreach ($segment as $logic => $condition) {
                $sql .= $sql ? ' '. $logic .' ' : '';

                if ($condition instanceOf static) {
                    $sql .= '('. $condition->buildSqlConditions($type) .')';
                } elseif (is_string($condition)) {
                    $sql .= $condition;
                } elseif (is_array($condition)) {
                    extract($condition);

                    if ($comparison == 'IN' || $comparison == 'NOT IN') {
                        $sql .= $this->quoteColumn($column) .' '. $comparison .'('. $this->escapeValues($value) .')';
                    } elseif ($comparison == 'BETWEEN' || $comparison == 'NOT_BETWEEN') {
                        $sql .= $this->quoteColumn($column) .' '. $comparison .' '. $this->escapeValue($value[0]) .' AND '. $this->escapeValue($value[1]);
                    } elseif ($comparison == 'IS NULL' || $comparison == 'IS NOT NULL') {
                        $sql .= $this->quoteColumn($column) .' '. $comparison;
                    } else {
                        $sql .= $this->quoteColumn($column) .' '. $comparison .' '. $this->escapeValue($value);
                    }
                }
            }
        }

        return $sql;
    }

    /**
     * 
     * 
     * @param string $sql 
     * @param array $builders
     * @return string
     */
    protected function buildSql($sql, array $builders = ['joins', 'wheres', 'groupings', 'havings', 'orderings', 'limit', 'offset'])
    {
        foreach ($builders as $builder) {
            if ( ! empty($this->{$builder})) {
                $sql = call_user_func([$this, 'buildSql'. ucfirst($builder)], $sql);
            }
        }

        return $sql;
    }

}

class Expression
{
    /**
     * @var string
     */
    protected $value;

    /**
     * 
     * 
     * @param string $value 
     * @return void
     */
    public function __construct($value)
    {
        $this->value = $value;
    }

    /**
     * 
     * 
     * @return string
     */
    public function __toString()
    {
        return $this->value;
    }

}