<?php

class Db {
    
    /** @var (object) Объект представляющий соединение с сервером */
    protected $link;
    
    /** @var (string) Хост */                                
    protected $host;
    
    /** @var (string) Имя пользователя */   
    protected $user;                

    /** @var (string) Пароль */
    protected $password;
    
    /** @var (string) Имя базы данных */
    protected $db;
    
    /**
     * Construct
     * Получает параметры соединения и устанавливает соединение с сервером
     * Устанавливает кодировку соединения
     * @param (string) $host хост
     * @param (string) $user имя пользователя
     * @param (string) $password пароль
     * @param (string) $db имя базы данных
     * @param (string) $charset кодировка соединения
     */
    
    public function __construct($host, $user, $password, $db, $charset)
    {

        $this->host = $host;
        $this->user = $user;
        $this->password = $password;
        $this->db = $db;
       
        
        $this->_connect();
        
        if (!$this->link)
        {
            
            throw new Exception('Can not connect to mysql server:<br />Error code: ' . mysqli_connect_errno() . '<br />' . mysqli_connect_error());
            
        }
        
        // Set charset
        $this->query("SET NAMES " . $charset);
        
    }
    
    /**
     * Подключение к серверу
     * @return (void)
     */
    
    protected function _connect() 
    {
        
        $this->link = mysqli_connect($this->host, $this->user, $this->password, $this->db);
        
    }
    
    /**
     * Закрытие подключения
     * @return (boolean)
     */
    
    public function close()
    {
        
        return mysqli_close($this->link);
        
    }
 
    /**
     * Смена базы данных
     * @param (string) $name Имя базы данных
     * @return (boolean)
     */
 
    public function selectDb($name)
    {
        
        return mysqli_select_db($this->link, $name);
        
    }
 
    /**
     * Осуществляет запрос к базе данных
     * @param (string) $query Строка запроса
     * @return (mixed)
     */
 
    public function query($string)
    {
        
        $query = mysqli_query($this->link, $string);
        
        if (!$query)
        {
            
            throw new Exception($string . '<br />Db error. Wrong query.<br />Error code: ' . $this->link->errno . '<br />Error: ' . $this->link->error);
            
        }
        
        return $query;
        
    }
    
    /**
     * Экранирует специальные символы в строке для использования в SQL выражении, берет текущую кодировку соединения
     * @param (string) $string строка для обработки
     * @return (string)
     */
    
    public function escapeString($string)
    {
        
        return mysqli_real_escape_string($this->link, $string);
        
    }
    
    /**
     * Получает все результирующие строки и возвращает массив со строками в виде 
     * ассоциативного массива и/или нумерованного массива. Доступно только с расширением mysqlnd.
     * @param (mysqli_result) $result результат запроса
     * @param (CONST) $resulttype: MYSQLI_ASSOC MYSQLI_NUM MYSQLI_BOTH тип результата выборки
     * @return (array)
     */
    
    public function fetchAll($result, $resulttype = MYSQLI_NUM)
    {
        
        return mysqli_fetch_all($result, $resulttype);
        
    }
    
    /**
     * Получает результирующую строку в виде ассоциативного массива
     * @param (mysqli_result) $result результат запроса
     * @return (array) 
     */
    
    public function fetchAssoc($result)
    {
        
        return mysqli_fetch_assoc($result);
        
    }
    
    /**
     * Получает ряд результата как нумерованный массив
     * @param (mysqli_result) $result результат запроса
     * @return (mixed)
     */
    
    public function fetchRow($result)
    {
        
        return mysqli_fetch_row($result);
        
    }
    
    /**
     * Возвращает из результата следующее поле
     * @param (mysqli_result) $result результат запроса
     * @return (object)
     */
    
    public function fetchField($result)
    {
        
        return mysqli_fetch_field($result);
        
    }
    
    /**
     * Освобождает память, определенную под результат
     * Вы должны всегда вызывать этот метод когда результат больше не нужен.
     * @param (mysqli_result) $result результат запроса
     * @return (void)
     */
    
    public function freeResult($result)
    {
        
        mysqli_free_result($result);
        
    }
    
    /**
     * Аналог mysql_result()
     * @param (string) $result результат запроса
     * @return (string)
     */
    
    public function result($result)
    {
        
        $result = $this->fetch_row($result);
        return $result[0];
        
    }
    
    /**
     * Получает число рядов в результирующей выборке
     * @param (mysqli_result) $result результат запроса
     * @return (int)
     */
     
    public function numRows($result)
    {
        
        return mysqli_num_rows($result);
        
    }
    
    /**
     * Получить количество полей в результате
     * @param (mysqli_result) $result результат запроса
     * @return (int)
     */
     
    public function numFields($result)
    {
        
        return mysqli_num_fields($result);
        
    }
    
    /**
     * Возвращает автоматически сгенерированный id, использованный в последнем запросе
     * @return (mixed)
     */
    
    public function insertId()
    {
        
        return $this->link->insert_id;
        
    }
    
    /**
     * Возвращает число строк, затронутых последним запросом
     * @return int
     */
    
    public function affectedRows()
    {
        
        return $this->link->affected_rows;
        
    }
    
}