<?php
namespace jl {

	/**
	 * JLayer
	 *
	 * Functions wrapper parent
	 *
	 * @package JohnCMS
	 * @subpackage JLayer
	 * @author Screamer
	 * @copyright 2013
	 */
	abstract class Functions {

		/**
		 * @var \jl\Template Templates handler
		 */
		protected $tpl;

		/**
		 * @var \jl\Db Db Object
		 */
		protected $db;

		/**
		 * @var int IP address
		 */
		private $ip;

		/**
		 * @var int IP address via PROXY server
		 */
		private $ip_via_proxy;

		/**
		 * @var string User-Agent
		 */
		private $user_agent;

		/**
		 * @var boolean Is mobile browser?
		 */
		private $is_mobile;

		/**
		 * @var array System settings
		 */
		private $system_settings;

		/**
		 * Constructor
		 *
		 * @param Template $tpl Templates handler
		 * @param Db       $db  Db Object
		 * @param int       $ip                IP address
		 * @param int       $ip_via_proxy      IP address via PROXY server
		 * @param string    $user_agent        User-Agent
		 * @param boolean   $is_mobile         Is mobile browser?
		 * @param array     $system_settings   System settings
		 *
		 * @return \jl\Functions
		 */
		public function __construct(Template $tpl, Db $db, $ip, $ip_via_proxy, $user_agent, $is_mobile, array $system_settings) {
			$this->tpl = $tpl;
			$this->db = $db;
			$this->ip = intval($ip);
			$this->ip_via_proxy = intval($ip_via_proxy);
			$this->user_agent = trim($user_agent);
			$this->is_mobile = (bool) $is_mobile;
			$this->system_settings = $system_settings;
		}

		/**
		 * Get system settings
		 *
		 * @param string $key  Name of Item (if empty, returns all array)
		 *
		 * @throws \InvalidArgumentException
		 * @return mixed
		 */
		public function get_settings($key = '') {
			if (!is_string($key)) {
				throw new \InvalidArgumentException('Invalid type of second argument. String expect.');
			}

			if (!empty($key)) {
				return !array_key_exists($key, $this->system_settings) ? null : $this->system_settings[$key];
			} else {
				return $this->system_settings;
			}
		}

		/**
		 * Returns IP address
		 *
		 * @param bool $format
		 *
		 * @return int|string
		 */
		public function get_ip($format = false) {
			return $format ? long2ip($this->ip) : $this->ip;
		}

		/**
		 * Returns IP address via PROXY server
		 *
		 * @param boolean $format Format address?
		 *
		 * @return int|string
		 */
		public function get_ip_via_proxy($format = false) {
			return $format ? long2ip($this->ip_via_proxy) : $this->ip_via_proxy;
		}

		/**
		 * Returns User-Agent
		 *
		 * @return string
		 */
		public function get_user_agent() {
			return $this->user_agent;
		}

		/**
		 * Is browser mobile?
		 *
		 * @return boolean
		 */
		public function is_mobile() {
			return $this->is_mobile;
		}

		/**
		 * Anti flood
		 * Check time interval between messages
		 * Returns time in seconds. If return false, posting is allowed.
		 *
		 * @return boolean|int
		 */
		public abstract function anti_flood();

		/**
		 * Panel with bb-codes
		 * @param string $form  Name of form
		 * @param string $field Name of field
		 *
		 * @return string
		 */
		public abstract function bb_panel($form, $field);

		/**
		 * Check captcha
		 *
		 * @param string $code Code for check
		 *
		 * @return boolean
		 */
		public abstract function check_captcha($code);

		/**
		 * Filter string
		 *
		 * @param string $string String for handle
		 *
		 * @return string
		 */
		public abstract function check_in($string);

		/**
		 * Handle string before output
		 *
		 * @param string $string  String for handle
		 * @param int    $nl      Handling new lines: 0 - no handle; 1 - translate to <br />; 2 - translate to spaces;
		 * @param int    $bb_tags Handling bb-codes: 0 - no handle; 1 - handle; 2 - cut;
		 * @param int    $smileys Handling smileys: 0 - no handle; 1 - handle;
		 *
		 * @return string
		 */
		public abstract function check_out($string, $nl = 0, $bb_tags = 0, $smileys = 0);

		/**
		 * Prepare confirm message to display
		 * Returns html form with two buttons: OK and Cancel;
		 *
		 * @param string $title   Title of form
		 * @param string $message Message
		 * @param string $ok      OK button value
		 * @param string $cancel  Cancel button value
		 * @param string $action  Form action: url to handler script
		 *
		 * @return string
		 */
		public abstract function confirm_message($title, $message, $ok, $cancel, $action = '');

		/**
		 * Handle time. Returns timestamp in days or hours.
		 *
		 * @param int $timestamp Timestamp
		 *
		 * @return string
		 */
		public abstract function count_time($timestamp);

		/**
		 * Translate timestamp to date (d.m.Y / H:i:s)
		 * Date with time-shifts, which defined in system and user settings!
		 *
		 * @param  int $timestamp Timestamp
		 *
		 * @return string
		 */
		public abstract function display_date($timestamp);

		/**
		 * Display some message
		 * @param array|string $message Message(s)
		 * @param int          $level   Level: 0 - notice; 1 - warning or error
		 *
		 * @return string
		 */
		public abstract function display_message($message, $level = 0);

		/**
		 * Prepare user data for display
		 *
		 * @param int|array $user Data of user
		 * @param boolean   $last_visit Enable last visit
		 * @param boolean   $status     Disable status
		 * @param boolean   $ips        Disable ip addresses and user-agent
		 * @param boolean   $ip_history Enable link to IP Addresses history
		 * @param string    $header     Display near nickname
		 * @param string    $body       Display under nickname
		 * @param string    $sub        Display at sub place
		 * @param string    $footer     Display at footer
		 *
		 * @return string
		 */
		public abstract function display_user(
			$user,
			$last_visit = false,
			$status = true,
			$ips = true,
			$ip_history = true,
			$header = '',
			$body = '',
			$sub = '',
			$footer = ''
		);

		/**
		 * Translate cyrillic chars to latin equivalents.
		 *
		 * @param string $string String for handle
		 *
		 * @return string
		 */
		public abstract function do_translit($string);

		/**
		 * Generates headers that force a download to happen
		 *
		 * @param   string $filename Filename
		 * @param   mixed  $data     the data to be downloaded
		 * @param   string $mime     Mime-Type of file
		 *
		 * @return  mixed
		 */
		public abstract function force_download($filename = '', $data = '', $mime = 'application/octet-stream');

		/**
		 * Parse smiles in string
		 *
		 * @param string  $string String for handle
		 * @param boolean $adm    Use admins smileys?
		 *
		 * @return string
		 */
		public abstract function do_smileys($string, $adm = false);

		/**
		 * Untranslit string
		 *
		 * @param string $string String for handle
		 *
		 * @return string
		 */
		public abstract function do_untranslit($string);

		/**
		 * Get captcha image and return html code for display.
		 * Also sets code on image to _SESSION['code']
		 *
		 * @param string $field_name Name of input field
		 * @param string $error      Error message if exists
		 *
		 * @return string
		 */
		public abstract function get_captcha($field_name, $error = '');

		/**
		 * Get user by id or nickname
		 * Returns empty array for non-exists user
		 * @param int|string $id ID or nickname of user
		 * @param string     $by Search field: id; nickname;
		 *
		 * @return array
		 */
		public abstract function get_user($id, $by = 'id');

		/**
		 * IP address validation
		 *
		 * @param string $ip IP address
		 *
		 * @return boolean
		 */
		public abstract function valid_ip($ip);

		/**
		 * Set time of last post
		 *
		 * @param $user_id
		 *
		 * @return boolean
		 */
		public abstract function set_last_post_time($user_id);

	}
}