<?php
namespace jl {

	use jl\exception\Exception;
	use jl\exception\InvalidArgumentException;

	/**
	 * JLayer
	 * JohnCMS core wrapper
	 * @package JohnCMS
	 * @subpackage JLayer
	 * @author Screamer
	 * @copyright 2013
	 */

	class JLayer {

		/**
		 * @var Users Users handler
		 */
		public $user;

		/**
		 * @var Functions Functions wrapper
		 */
		public $functions;

		/**
		 * @var Db Object of Db class, which provides access to MySQL
		 */
		private $db;

		/**
		 * @var string Prefix for database tables
		 */
		private $table_prefix;

		/**
		 * @var string Path to directory with Models
		 */
		private $models_dir;

		/**
		 * @var string Language ISO code
		 */
		private $lng;

		/**
		 * @var array Loaded models
		 */
		private $models = array();

		/**
		 * @var array Loaded plugins
		 */
		private $plugins = array();

		/**
		 * Construct
		 *
		 * @param Db        $db                Db object
		 * @param Users     $user              Users handler
		 * @param Functions $functions         Functions wrapper
		 * @param string    $table_prefix      Prefix for database tables
		 * @param string    $models_dir        Path to directory with Models
		 * @param string    $lang              Language ISO code
		 *
		 * @throws exception\Exception
		 * @return JLayer
		 */
		public function __construct(
			Db $db,
			Users $user,
			Functions $functions,
			$table_prefix,
			$models_dir,
			$lang
		) {
			$this->db = $db;
			$this->user = $user;
			$this->functions = $functions;
			$this->table_prefix = trim($table_prefix);
			$this->lng = strval($lang);
			if (empty($this->table_prefix)) {
				throw new Exception('Invalid value of tables prefix');
			}
			if (!is_dir($models_dir)) {
				throw new Exception('Wrong path to directory of models: ' . $models_dir);
			}
			$this->models_dir = $models_dir;
		}

		/**
		 * Load Model
		 *
		 * @param string $name Name of model
		 *
		 * @throws exception\Exception
		 * @throws exception\InvalidArgumentException
		 * @return object
		 */
		public function model($name) {
			if (!is_string($name)) {
				throw new InvalidArgumentException('name', 'string');
			}
			if (!array_key_exists($name, $this->models)) {
				$model = $this->models_dir . $name . '.php';
				if (is_file($model)) {
					require $model;
					$class = 'Model_' . ucwords($name);
					if (class_exists($class)) {
						$this->models[$name] = new $class($this->db, $this->get_tables_prefix());
					} else {
						throw new Exception('Unable to load Model. Class ' . $class . ' is not exists.');
					}
				} else {
					throw new Exception('Unable to load Model. File ' . $model . ' is not exists.');
				}
			}
			return $this->models[$name];
		}

		/**
		 * Load plugin
		 *
		 * @param string $name Name of plugin
		 *
		 * @throws exception\Exception
		 * @throws exception\InvalidArgumentException
		 * @return object
		 */
		public function plugin($name) {
			if (!is_string($name)) {
				throw new InvalidArgumentException('name', 'string');
			}
			if (!array_key_exists($name, $this->plugins)) {
				$plugin_path = JL_DIR . 'jl' . DIR_SEP . 'plugins' . DIR_SEP . $name . DIR_SEP;
				$plugin = $plugin_path . $name . '.php';
				if (is_file($plugin)) {
					$class = '\jl\plugins\\' . ucwords($name);
					require $plugin;
					if (!class_exists($class)) {
						throw new Exception('Unable to load plugin. Class "' . $class . '" is not exists.');
					} else {
						if (is_dir($plugin_path . 'view')) {
							$tpl_obj = new Template($plugin_path . 'view', 'php');
						} else {
							$tpl_obj = null;
						}
						if (is_file($plugin_path . 'lang' . DIR_SEP . $this->lng . '.json')) {
							$lng_obj = new Language($plugin_path . 'lang' . DIR_SEP . $this->lng . '.json');
						} else {
							$lng_obj = null;
						}
						$this->plugins[$name] = new $class($this->db, $tpl_obj, $this, $lng_obj);
					}
				} else {
					throw new Exception('Unable to load plugin. File "' . $plugin . '" is not exists.');
				}
			}
			return $this->plugins[$name];
		}

		/**
		 * Load helper
		 *
		 * @param string $name Name of helper
		 *
		 * @return boolean
		 */
		public function helper($name) {
			$path = JL_MODULE_DIR . 'helpers' . DIR_SEP . $name . '.php';
			if (is_file($path)) {
				require $path;
				return true;
			} else {
				return false;
			}
		}

		/**
		 * Return prefix for database tables
		 *
		 * @return string
		 */
		public function get_tables_prefix() {
			return $this->table_prefix;
		}

		/**
		 * Load configuration file
		 *
		 * @param string $name Name of file
		 * @param string $from Load From: module|plugin
		 *
		 * @throws exception\Exception
		 * @return array
		 */
		public function load_config($name = '', $from = 'module') {
			if ($from == 'plugin') {
				$name = explode('/', $name);
				$name = sizeof($name) == 2 ? $name[0] . '/config/' . $name[1] : implode('/', $name);
			}
			$path = $from == 'module' ? JL_MODULE_DIR . 'config' : JL_DIR . 'jl' . DIR_SEP . 'plugins' ;
			$file = realpath($path . DIR_SEP . $name . '.json');
			if ($file) {
				$config = json_decode(file_get_contents($file), true);
				if (!is_array($config)) {
					throw new Exception('Wrong format of configuration file "' . $file . '".');
				}
				return $config;
			} else {
				throw new Exception('Configuration file "' . $path . DIR_SEP . $name . '.json' . '" is not exists.');
			}
		}

	}
}