<?php
namespace jl {

	use jl\exception\Exception;
	use jl\exception\InvalidArgumentException;

	/**
	 * JLayer
	 *
	 * Templates Handler
	 *
	 * @package JohnCMS
	 * @subpackage JLayer
	 * @author Screamer
	 * @copyright 2013
	 */

	class Template {

		/**
		 * @var string Path to templates
		 */
		protected $path;

		/**
		 * @var string Extension of templates
		 */
		protected $ext;

		/**
		 * @var string Title of page
		 */
		protected $title;

		/**
		 * @var array Data for output
		 */
		protected $output = array();

		/**
		 * Construct
		 *
		 * @param string $path Path to templates
		 * @param string $ext  Extension of templates (without dot)
		 *
		 * @throws exception\Exception
		 * @throws exception\InvalidArgumentException
		 * @return \jl\Template
		 */
		public function __construct($path, $ext) {
			if (!is_string($path)) {
				throw new InvalidArgumentException('path', 'string');
			}
			if (!is_string($ext)) {
				throw new InvalidArgumentException('ext', 'string');
			}

			$path = rtrim($path , '/\\') . DIR_SEP;
			if (is_dir($path)) {
				$this->path = $path;
			} else {
				throw new Exception(
					'Unable to set path to templates. Directory "'
					. $path
					. '" is not exists.'
				);
			}
			$ext = ltrim($ext, '.');
			$this->ext = '.' . $ext;

		}

		/**
		 * Get title of page
		 *
		 * @return string|null
		 */
		public function get_title() {
			return $this->title;
		}

		/**
		 * Load template
		 *
		 * @param string $name Name of template
		 * @param array  $data Array with variables for template
		 *
		 * @throws exception\InvalidArgumentException
		 * @return string
		 */
		public function load($name, array $data = array()) {
			if (!is_string($name)) {
				throw new InvalidArgumentException('name', 'string');
			}
			ob_start();
			if (is_file($this->path . $name . $this->ext)) {
				if (!empty($data)) {
					extract($data, EXTR_SKIP);
				}
				require $this->path . $name . $this->ext;
			} else {
				echo __CLASS__ . '::Unable to load template. Template "' . $this->path . $name . $this->ext . '" is not exists.';
			}
			$contents = ob_get_contents();
			ob_end_clean();
			return $contents;
		}

		/**
		 * Display data for output
		 *
		 * @return void
		 */
		public function output() {
			foreach ($this->output as $item) {
				echo $item;
			}
		}

		/**
		 * Set data for output
		 *
		 * @param string $data     Data
		 * @param int    $priority Priority of data
		 *
		 * @throws exception\InvalidArgumentException
		 * @return void
		 */
		public function set_output($data, $priority = null) {
			if (!is_string($data)) {
				throw new InvalidArgumentException('data', 'string');
			}
			if (!is_null($priority) && !is_integer($priority)) {
				throw new InvalidArgumentException('priority', 'null, integer');
			}
			$priority = is_null($priority) ? sizeof($this->output) : $priority;
			$this->output[$priority] = $data;
		}

		/**
		 * Set title of page
		 *
		 * @param string $title Title
		 *
		 * @throws exception\InvalidArgumentException
		 * @return void
		 */
		public function set_title($title) {
			if (!is_string($title)) {
				throw new InvalidArgumentException('title', 'string');
			}
			$this->title = trim($title);
		}

	}
}
