<?php
/**
 * Класс для поска/замены всех возможных вариантов запрещённых сайтов
 * @author Rakovskiy Andrey <rakovskiy28@mail.ru>
 * @link http://vk.com/rakovskiy28
 * @version 1.0
 *
 *
 * Пример использования
 * <pre>
 *      $obj = new RBlackSites();
 *
 *      $obj->sites = [
 *          [
 *              'host' => 'google',
 *              'domain' => 'com',
 *          ],
 *          [
 *              'host' => 'youtube',
 *              'domain' => 'com',
 *          ],
 *      ];
 *
 *      $obj->mask = [
 *          'http://{host}.{domain}',
 *          'https://{host}.{domain}',
 *          '{host}.{domain}',
 *          '{host},{domain}',
 *          '{host}_{domain}',
 *      ];
 *
 *      $obj->replace = '***SPAM***';
 *      $obj->setText('Привет, заходи на мой сайт google.com');
 *
 *      echo $obj->getText();
 * </pre>
 *
 * В пиведенном выше примере сайты в тексте вида
 *
 * http://google.com
 * https://google.com
 * google.com
 * google,com
 * google_com
 *
 * будут заменены на ***SPAM***
 *
 * ВНИМАНИЕ!
 * $obj->setText($text) должен быть объявлен после указания $obj->sites и $obj->mask
 *
 * В данном классе есть метод для проверки наличия запрещённых сайтов в тексте
 * для этого используется метод $obj->has()
 */

class RBlackListSites {

    /**
     * @var array Список запрещённых сайтов
     */
    public $sites = [];

    /**
     * @var array Варианы замены запрещённых сайтов
     */
    public $mask = [];

    /**
     * @var string На что будем заменять сайт
     */
    public $replace;

    /**
     * @var array Массив со всеми возможными вариантами замены для каждого сайта
     */
    private $arraySites = [];

    /**
     * @var string Исходный текст который подлежит дальнейшей проверки/замены запрещённых сайтов
     */
    private $text;

    /**
     * @param string $text Текст для поиска/замены запрещённых сайтов
     */
    public function setText($text){
        $this->arraySites = $this->arraySites();
        $this->text = $text;
    }

    /**
     * Возвращает форматированный от запрещённых сайтов текст
     * @return string
     */
    public function getText()
    {
        return str_ireplace(array_keys($this->arraySites), $this->arraySites, $this->text);
    }

    /**
     * Проверяем на наличие запрещённых сайтов в тексте
     * @return boolean
     */
    public function has(){
        return $this->text != $this->replace();
    }

    /**
     * Возвращаем массив возможных вариантов замены для каждого сайта
     * @return array
     */
    private function arraySites(){
        $sites = [];

        foreach ($this->sites as $site){
            foreach ($this->mask as $mask){
                $str = strtr($mask, [
                    '{host}' => $site['host'],
                    '{domain}' => $site['domain']
                ]);
                $sites[$str] = $this->replace;
            }
        }

        return $sites;
    }
}