<?php
/*
 * Copyright (C) 2015 Jahak
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
class Logs
{

    /**
     * @var int кол-во файлов для записи
     */
    private $countFiles = 10;

    /**
     * @var array массив данных
     */
    private $data;

    /**
     * @var string разделитель
     */
    public $glue = ':|:';

    /**
     * @var string расширение файлов
     */
    private $suffix = '.jahak';

    /**
     * @var int время для записи в один файл
     */
    private $time = 300;

    public function __construct()
    {
        // данные
        $this->data = [
            // дата и время
            time(),
            // user agent
            (filter_has_var(INPUT_SERVER, 'HTTP_USER_AGENT') ? filter_input(INPUT_SERVER, 'HTTP_USER_AGENT', FILTER_SANITIZE_STRING) : 'Unknown'),
            // ip
            (filter_has_var(INPUT_SERVER, 'REMOTE_ADDR') ? filter_input(INPUT_SERVER, 'REMOTE_ADDR') : 'Unknown')
        ];
        // недопускает для записи повторных ip и user agent
        if ( !in_array($this->getUaIp(), array_keys($this->read())) )
        {
            // идет запись
            $this->write();
        }
    }

    /**
     * user agent glue ip
     * @return string
     */
    public function getUaIp()
    {
        // Mozilla/5.0 (Windows NT 6.1; rv:41.0) Gecko/20100101 Firefox/41.0:|:127.0.0.1
        return $this->data[1] . $this->glue . $this->data[2];
    }

    /**
     * Выводит общую информацию
     * @return string
     */
    public function info()
    {
        return sprintf('Пользователей: %s<br>Создано файлов: %s<br>Общий размер файлов: %s байт', self::numberFormat(count($this->read())), count($this->getFiles()), self::numberFormat(array_sum(array_map('filesize', $this->getFiles()))));
    }

    /**
     * Формирует число с разделением групп
     * @param int $number
     * @return string
     */
    private static function numberFormat($number)
    {
        return number_format($number, null, null, ' ');
    }

    /**
     * Возвращает список файлов
     * @return array
     */
    private function getFiles()
    {
        return glob('*' . $this->suffix);
    }

    /**
     * @param bool $bool
     * @return bool|string
     */
    private function endFile($bool = false)
    {
        return ($bool ? ((time() - $this->time) > ((int) basename(end($this->getFiles()), $this->suffix))) : end($this->getFiles()));
    }

    /**
     * Выводит все ip и user agent
     * @return array
     */
    public function read()
    {
        $return = [];
        foreach ( array_filter(explode(PHP_EOL, implode(PHP_EOL, array_map('file_get_contents', $this->getFiles())))) as $value )
        {
            $explode = explode($this->glue, $value);
            $shift = array_shift($explode);
            $return[implode($this->glue, $explode)] = $shift;
        }
        return $return;
    }

    /**
     * Пишет полученный массив данных в файл
     */
    private function write()
    {
        if ( count($this->getFiles()) < $this->countFiles && $this->endFile(true) )
        {
            $filename = time() . $this->suffix;
        }
        else
        {
            $filename = $this->endFile();
        }
        return file_put_contents($filename, implode($this->glue, $this->data) . PHP_EOL, FILE_APPEND | LOCK_EX);
    }

}
