<?php
declare(strict_types=1);

namespace Qa\Actions;

use Johncms\Notifications\Notification;
use Qa\Context;

class Answer extends AbstractAction
{
    public function __construct(Context $context)
    {
        parent::__construct($context);
        $this->title = 'Новый ответ';
    }

    public function invoke() : void
    {
        $id = intval($this->context->routeParam() ?: 0);
        if ($id <= 0) {
            $this->resultNotFound();
        }

        $questionsEntity = $this->context->questionsEntity();
        $question = $questionsEntity->getById($id);
        if (empty($question['id'])) {
            $this->resultNotFound();
        }
        $this->backUrl = "/qa/show/$id";

        $flood = $this->context->tools()->antiflood();
        if ($flood) {
            $this->resultTooManyRequests($flood);
        }

        $answersEntity = $this->context->answersEntity();
        if (!$answersEntity->canAnswer()) {
            $this->resultForbidden();
        }

        $answer = [
            'question_id' => $id
        ];

        $errors = [];
        if (isset($_POST['submit'])) {
            $this->processPostData($answer);
            $errors = $this->validateAnswer($answer);
            if (empty($errors)) {
                $answerId = $answersEntity->insert($answer);
                if ($answer['attachment_name'] && $this->hasFile()) {
                    $this->finishUploadFile($answerId);
                }
                $this->notifyAuthorIfNecessary($question, $answerId);
                $this->updateLastPost();
                $_SESSION['alert'] = [
                    'type' => 'alert-success',
                    'content' => 'Ответ добавлен. <a class="alert-link" href="#a' . $answerId . '">Перейти</a>'
                ];
                header('Location: ' . $this->backUrl);
                return;
            }
        }

        $this->context->navChain()->add(
                htmlspecialchars($question['title']),
                $this->backUrl);
        $this->render('qa::answer', [
            'answer' => $answer,
            'errors' => $errors
        ]);
    }

    /**
     * Заполняет данные из формы в модель
     * @param array $answer
     * @return void
     */
    protected function processPostData(array &$answer) : void
    {
        $answer['text'] = trim($_POST['text'] ?? '');
        if (empty($answer['attachment_name'])) {
            $answer['attachment_name'] = false;
        }
        if ($this->hasFile()) {
            $info = pathinfo( $_FILES['file']['name']);
            $filename = mb_substr($info['filename'] ?? 'file', 0, 80);
            $ext = mb_substr($info['extension'] ?? '', 0, 10);
            $answer['attachment_name'] = "$filename.$ext";
        }
    }

    /**
     * Завершает выгрузку файла на сервер
     * @param int $id
     * @return void
     */
    protected function finishUploadFile(int $id) : void
    {
        move_uploaded_file($_FILES['file']['tmp_name'], QA_UPLOAD_PATH . "a_$id.dat");
    }

    protected function validateAnswer(array &$answer) : array
    {
        $errors = [];
        if (empty($answer['text']))
            $errors[] = 'Вы не ввели ответ';
        elseif (mb_strlen($answer['text']) > 10000)
            $errors[] = 'Длина ответа не должна превышать 10000 символов';

        $config = $this->context->config();
        $maxFileSizeInBytes = $config['flsz'] * 1024;
        if (!$errors && $this->hasFile() && ($_FILES['file']['size'] >= $maxFileSizeInBytes)) {
            $errors[] = 'Файл превышает допустимый размер в ' . (format_size($maxFileSizeInBytes));
        }

        return $errors;
    }

    private function notifyAuthorIfNecessary(array &$question, int $answerId) : void
    {
        $currentUserId = $this->context->user()->id;
        if ($question['notify'] && ($currentUserId != $question['author_id'])) {
            (new Notification())->create(
                [
                    'module' => 'qa',
                    'event_type' => 'new_answer',
                    'user_id' => $question['author_id'],
                    'sender_id' => $currentUserId,
                    'entity_id' => $answerId,
                    'fields' => [
                        'id' => $answerId,
                        'title' => htmlspecialchars($question['title'])
                    ]
                ]
            );
        }
    }

    protected function hasFile() : bool
    {
        if (empty($_FILES['file'])) {
            return false;
        }
        $size = ($_FILES['file']['size'] ?? 0);
        return $size > 0;
    }
}
